import 'package:flutter/material.dart';
import 'package:flutter_spinkit/flutter_spinkit.dart';
import 'package:get/get.dart';
import 'package:viserpay/core/helper/string_format_helper.dart';
import 'package:viserpay/core/utils/dimensions.dart';
import 'package:viserpay/core/utils/my_color.dart';
import 'package:viserpay/core/utils/my_icon.dart';
import 'package:viserpay/core/utils/my_images.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/style.dart';
import 'package:viserpay/data/model/reward/reward_response_model.dart';
import 'package:viserpay/view/components/image/custom_svg_picture.dart';

class RewardCard extends StatelessWidget {
  final VoidCallback onTap;
  final Reward reward;
  final bool isLoading;
  final bool isUsed;
  final bool isDisable;
  const RewardCard({
    super.key,
    required this.reward,
    required this.onTap,
    this.isLoading = false,
    this.isUsed = false,
    this.isDisable = false,
  });

  @override
  Widget build(BuildContext context) {
    return Container(
      padding: EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: 5),
      decoration: BoxDecoration(color: MyColor.colorWhite, borderRadius: BorderRadius.circular(Dimensions.largeRadius)),
      child: Row(
        children: [
          rewardImage(),
          const SizedBox(width: Dimensions.space10),
          Expanded(
            flex: 2,
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(reward.name ?? "", style: regularDefault.copyWith()),
                Text(reward.description ?? "", style: lightDefault.copyWith()),
                Text("${StringConverter.formatNumber(reward.requiredPoints ?? '0', precision: 0)} ${MyStrings.points.tr}", style: semiBoldDefault.copyWith(color: MyColor.colorRed)),
              ],
            ),
          ),
          const SizedBox(width: Dimensions.space10),
          isUsed
              ? Text(MyStrings.claimed.tr, style: semiBoldDefault.copyWith(color: MyColor.greenSuccessColor))
              : isDisable
                  ? Container(
                      padding: EdgeInsets.all(5),
                      decoration: BoxDecoration(color: MyColor.colorBlack.withValues(alpha: 0.2), shape: BoxShape.circle),
                      child: Icon(Icons.lock, color: MyColor.colorBlack, size: 18),
                    )
                  : Expanded(
                      flex: 1,
                      child: InkWell(
                        onTap: (isLoading || isDisable) ? null : onTap,
                        child: Container(
                          padding: EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: 8),
                          decoration: BoxDecoration(
                            color: MyColor.primaryColor,
                            borderRadius: BorderRadius.circular(14),
                          ),
                          child: Center(
                            child: isLoading ? SizedBox(height: 20, width: 20, child: const SpinKitCircle(color: MyColor.colorWhite, size: 18)) : Text(MyStrings.claim, style: semiBoldDefault.copyWith(color: MyColor.colorWhite)),
                          ),
                        ),
                      ),
                    )
        ],
      ),
    );
  }

  Widget rewardImage() {
    String transactionType = reward.transactionType.toString().toLowerCase();

    Widget imageWidget;

    switch (transactionType) {
      case "send_money":
        imageWidget = const CustomSvgPicture(
          image: MyIcon.sendMoney,
          color: MyColor.sendMoneyBaseColor,
          height: 24,
          width: 24,
        );
        break;
      case "mobile_recharge":
        imageWidget = const CustomSvgPicture(
          image: MyIcon.mobileRecharge,
          color: MyColor.rechargeBaseColor,
          height: 24,
          width: 24,
        );
        break;
      case "cash_out":
        imageWidget = const CustomSvgPicture(
          image: MyIcon.cashout,
          color: MyColor.cashoutBaseColor,
          height: 24,
          width: 24,
        );
        break;
      case "make_payment":
        imageWidget = const CustomSvgPicture(
          image: MyIcon.cart,
          color: MyColor.paymentBaseColor,
          height: 24,
          width: 24,
        );
        break;
      case "bank_transfer":
        imageWidget = const CustomSvgPicture(
          image: MyIcon.bank,
          color: MyColor.bankTransferBaseColor,
          height: 24,
          width: 24,
        );
        break;
      case "utility_bill":
        imageWidget = const CustomSvgPicture(
          image: MyIcon.paybill,
          color: MyColor.paybillBaseColor,
          height: 32,
          width: 32,
        );
        break;
      case "donation":
        imageWidget = const CustomSvgPicture(
          image: MyIcon.donation,
          color: MyColor.donationBaseColor,
          height: 26,
          width: 26,
        );
        break;
      case "add_money":
        imageWidget = const CustomSvgPicture(image: MyIcon.addMoney, color: MyColor.addMoneyBaseColor, height: 24, width: 24);
        break;
      case "create_voucher":
        imageWidget = Image.asset(MyImages.voucherImage, color: Colors.pink, height: 24, width: 24);
        break;
      case "request_money":
        imageWidget = Image.asset(MyImages.reqMoneyImage, color: Colors.teal, height: 24, width: 24);
        break;
      case "air_time":
        imageWidget = Image.asset(MyImages.airTimeImage, color: Colors.lightBlue, height: 24, width: 24);
        break;
      case "invoice":
        imageWidget = const CustomSvgPicture(image: MyIcon.invoiceIcon, color: Colors.purpleAccent, height: 24, width: 24);
        break;
      case "auto_payment":
        imageWidget = Image.asset(MyIcon.autoPayIcon, color: Colors.pink, height: 24, width: 24);
        break;
      case "balance_add":
        imageWidget = const CustomSvgPicture(image: MyIcon.addMoney, color: MyColor.addMoneyBaseColor, height: 24, width: 24);
        break;
      case "deposit":
        imageWidget = const CustomSvgPicture(image: MyIcon.addMoney, color: MyColor.addMoneyBaseColor, height: 24, width: 24);
        break;
      case "referral_bonus":
        imageWidget = const CustomSvgPicture(image: MyIcon.addMoney, color: MyColor.addMoneyBaseColor, height: 24, width: 24);
        break;
      default:
        imageWidget = Image.asset(
          MyImages.reqMoneyImage,
          height: 20,
          width: 20,
        );
    }

    return Container(
      padding: const EdgeInsets.all(7),
      decoration: BoxDecoration(color: MyColor.colorWhite, border: Border.all(color: MyColor.borderColor), borderRadius: BorderRadius.circular(Dimensions.mediumRadius)),
      child: imageWidget,
    );
  }
}
